///////////////////////////////////////////////////////////////////////////////////////
//
// MonoScan
//
// 27/04/07		Hounddog	Initial implementation
//

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Data;
using System.Windows.Forms;

namespace MonoScan
{
	public class ZoneArrowIndicator : ZoneIndicator
	{
		const int marginPer = 5;			// % of border height/width
		const int circleAngle = 270;		// degrees
		const int arrowRadiusPer = 5;		// % of circle radius
		const int arrowLengthPer = 94;		// % of circle radius

		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		public ZoneArrowIndicator()
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();

			range.Min = 0;
			range.Max = 255;

			normalRange.Min = 0;
			normalRange.Max = 0;			

			weight = 0;

			unit = null;

			points = null;

			value = 0;

			rawMode = false;

			UpdatePaint();

			SetStyle(ControlStyles.UserPaint, true);
			SetStyle(ControlStyles.AllPaintingInWmPaint, true);
			SetStyle(ControlStyles.DoubleBuffer, true);
		}		

		public string Caption
		{
			get { return caption; }

			set
			{
				if (value != caption)
				{
					caption = value;

					Invalidate();
				}
			}
		}

		protected string caption;

		public Range Range
		{
			get { return range; }

			set 
			{
				if (value.Min != range.Min || value.Max != range.Max)
				{
					if (value.Min >= value.Max)
						throw new ArgumentException("Invalid range");

					range = value;

					UpdatePaint();
				}
			}
		}

		protected Range range;

		/// <remarks>
		/// Set Min = Max to disable normal range. 
		/// Normal range can be wider than or equal to range.
		/// </remarks>
		public Range NormalRange
		{
			get { return normalRange; }

			set 
			{
				if (value.Min != normalRange.Min || value.Max != normalRange.Max)
				{
					if (value.Min > value.Max)
						throw new ArgumentException("Invalid range");

					normalRange = value;

					UpdatePaint();
				}
			}
		}

		protected Range normalRange;

		public int Weight
		{
			get { return weight; }

			set 
			{
				if (value != weight)
				{
					weight = value;

					UpdatePaint();
				}
			}
		}

		int weight;

		public string Unit
		{
			get { return unit; }

			set
			{
				if (value != unit)
				{
					unit = value;

					UpdatePaint();
				}
			}
		}

		protected string unit;

		/// <summary>
		/// Optional transformation curve applied to Value, Range, NormalRange.
		/// </summary>
		/// <remarks>
		/// The points must be ordered by X and must be monotone by Y.
		/// Minimum number of points 2. The more the points the better.
		/// </remarks>
		public Point[] Points
		{
			get { return points; }

			set
			{
				if (value == null || value.Length < 2)
					throw new ArgumentException("Invalid points array");

				int pointsDelta = value[value.Length - 1].Y - value[0].Y;

				for (int i = 1; i < value.Length; ++ i)
				{
					if (value[i].X - value[i - 1].X <= 0)
						throw new ArgumentException("Invalid points array");

					if ((value[i].Y - value[i - 1].Y) * pointsDelta < 0)
						throw new ArgumentException("Invalid points array");
				}

				points = value;
				this.pointsDelta = pointsDelta;

				UpdatePaint();
			}
		}
	
		protected Point[] points;
		protected int pointsDelta;

		public override int Value
		{
			get { return value; }

			set
			{
				if (value != this.value)
				{
					this.value = value;

					UpdatePaint();
				}
			}
		}

		protected int value;

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if( components != null )
					components.Dispose();
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			components = new System.ComponentModel.Container();
		}
		#endregion

		protected override void OnDoubleClick(EventArgs e)
		{
			rawMode = ! rawMode;

			UpdatePaint();
		}

		protected bool rawMode;

		protected virtual void UpdatePaint()
		{
			if (! rawMode)
			{
				if (points != null)
				{
					if (pointsDelta > 0)
					{
						rangePaint.Min = GetPointedValue(range.Min);
						rangePaint.Max = GetPointedValue(range.Max);

						normalRangePaint.Min = GetPointedValue(normalRange.Min);
						normalRangePaint.Max = GetPointedValue(normalRange.Max);
					}
					else
					{
						rangePaint.Min = GetPointedValue(range.Max);
						rangePaint.Max = GetPointedValue(range.Min);

						normalRangePaint.Min = GetPointedValue(normalRange.Max);
						normalRangePaint.Max = GetPointedValue(normalRange.Min);
					}

					valuePaint = GetPointedValue(value);
				}
				else
				{
					rangePaint = range;

					normalRangePaint = normalRange;

					valuePaint = value;
				}

				minRangeText = GetValueText(rangePaint.Min);
				maxRangeText = GetValueText(rangePaint.Max);

				if (normalRangePaint.Min != normalRangePaint.Max)
				{
					normalText = GetValueText(normalRangePaint.Min) + ".." + GetValueText(normalRangePaint.Max);

					normalValue = valuePaint == NotAvailableValue || valuePaint >= normalRangePaint.Min && 
						valuePaint <= normalRangePaint.Max;
				}
				else
				{
					normalText = null;

					normalValue = true;
				}
				
				if (valuePaint == NotAvailableValue)
				{
					valueText = "NA";
				}
				else if (valuePaint < rangePaint.Min || valuePaint > rangePaint.Max)
				{
					valueText = "OVF";
				}
				else if (unit != null)
				{
					valueText = GetValueText(valuePaint) + ' ' + unit;
				}
				else
					valueText = GetValueText(valuePaint);
			}
			else
			{
				rangePaint = range;

				normalRangePaint = normalRange;

				valuePaint = value;

				minRangeText = rangePaint.Min.ToString();
				maxRangeText = rangePaint.Max.ToString();

				if (normalRangePaint.Min != normalRangePaint.Max)
				{
					normalText = normalRangePaint.Min.ToString() + ".." + normalRangePaint.Max.ToString();

					normalValue = valuePaint == NotAvailableValue || valuePaint >= normalRangePaint.Min && 
						valuePaint <= normalRangePaint.Max;
				}
				else
				{
					normalText = null;

					normalValue = true;
				}
				
				if (valuePaint == NotAvailableValue)
				{
					valueText = "NA";
				}
				else if (valuePaint < rangePaint.Min || valuePaint > rangePaint.Max)
				{
					valueText = "OVF";
				}
				else
					valueText = value.ToString();
			}

			Invalidate();
		}

		protected override void OnPaint(PaintEventArgs pe)
		{
			pe.Graphics.SmoothingMode = SmoothingMode.AntiAlias;

			Rectangle clientRectangle = ClientRectangle;

			// 3D border & margines

			if (clientRectangle.Width < 2 || clientRectangle.Height < 2)
				return;

			Rectangle borderRectangle = new Rectangle(clientRectangle.Left + 1, clientRectangle.Top + 1, 
				clientRectangle.Width - 2, clientRectangle.Height - 2);

			using (Pen pen = new Pen(SystemColors.ControlLightLight))
			{
				pe.Graphics.DrawLine(pen, borderRectangle.Left, borderRectangle.Bottom, 
					borderRectangle.Left, borderRectangle.Top);

				pe.Graphics.DrawLine(pen, borderRectangle.Left, borderRectangle.Top,
					borderRectangle.Right, borderRectangle.Top);
			}

			using (Pen pen = new Pen(SystemColors.ControlDark))
			{
				pe.Graphics.DrawLine(pen, borderRectangle.Right, borderRectangle.Top,
					borderRectangle.Right, borderRectangle.Bottom);

				pe.Graphics.DrawLine(pen, borderRectangle.Right, borderRectangle.Bottom,
					borderRectangle.Left, borderRectangle.Bottom);
			}

			float marginHor = borderRectangle.Width * marginPer / 100;
			float marginVer = borderRectangle.Height * marginPer / 100;

			if (borderRectangle.Height  < 2 * marginVer || borderRectangle.Width < 2 * marginHor)
				return;

			RectangleF drawRectangle = new RectangleF(borderRectangle.Left + marginHor,
				borderRectangle.Top + marginVer, borderRectangle.Width - 2 * marginHor,
				borderRectangle.Height - 2 * marginVer);		

			// Caption

			float lineHeight = pe.Graphics.MeasureString("X", Font).Height;

			if (drawRectangle.Height < lineHeight)
				return;

			StringFormat stringFormat = new StringFormat();
			stringFormat.Alignment = StringAlignment.Center;
			stringFormat.FormatFlags = StringFormatFlags.NoWrap;

			RectangleF captionRectangle = new RectangleF(drawRectangle.Left, drawRectangle.Top,
				drawRectangle.Width, lineHeight);

			using (Brush brush = new SolidBrush(normalValue ? SystemColors.WindowText : Color.Red))
			{		
				pe.Graphics.DrawString(caption, Font, brush, captionRectangle, stringFormat);
			}		

			// Indicator surface

			if (drawRectangle.Bottom - captionRectangle.Bottom < 3 * lineHeight)
				return;

			const int alpha = (360 - circleAngle) / 2;

			float circleRadiusHor = drawRectangle.Width / 2;
			float circleRadiusVer = (drawRectangle.Bottom - captionRectangle.Bottom - marginVer -
				lineHeight) / (1 + 0.07F + (float) Math.Cos(DegreesToRadians(alpha)));

			float circleRadius = Math.Min(circleRadiusHor, circleRadiusVer);

			PointF circlePoint = new PointF(
				(drawRectangle.Left + drawRectangle.Right) / 2,
				(captionRectangle.Bottom + marginVer + drawRectangle.Bottom - circleRadius * 
				(1 + 0.07F + (float) Math.Cos(DegreesToRadians(alpha))) - lineHeight) / 2 + 
				circleRadius);

			GraphicsPath circleGraphicsPath = new GraphicsPath();

			RectangleF circleRectangle = new RectangleF(
				circlePoint.X - circleRadius,
				circlePoint.Y - circleRadius,
				2 * circleRadius,
				2 * circleRadius);

			circleGraphicsPath.AddArc(circleRectangle.Left, circleRectangle.Top, 
				circleRectangle.Width, circleRectangle.Height, 90 + alpha, circleAngle);

			circleGraphicsPath.CloseFigure();

			using (Brush brush = new SolidBrush(Color.White))
			{
				pe.Graphics.FillPath(brush, circleGraphicsPath);
			}

			if (normalRangePaint.Min != normalRangePaint.Max)
			{
				// Normal range surface

				float minNormalAngle = GetValueAngle(normalRangePaint.Min);
				float maxNormalAngle = GetValueAngle(normalRangePaint.Max);

				using (Brush brush = new SolidBrush(Color.PaleTurquoise))
				{
					pe.Graphics.FillPie(brush, circleRectangle.Left, circleRectangle.Top, 
						circleRectangle.Width, circleRectangle.Height,
						90 + alpha + minNormalAngle, maxNormalAngle - minNormalAngle);
				}
			}

			using (Pen circlePen = new Pen(SystemColors.ControlDark))
			{
				pe.Graphics.DrawPath(circlePen, circleGraphicsPath);
			}

			// Marks

			using (Pen pen = new Pen(Color.Black))
			{
				float circleInnerRadius1 = circleRadius * 0.88F;
				float circleInnerRadius2 = circleRadius * 0.94F;

				float deltaAngle = (float) (DegreesToRadians(circleAngle) / 20);
				float angle = (float) DegreesToRadians(alpha);

				for (int i = 0; i < 21; ++ i)
				{
					float circleInnerRadius = i % 2 == 0 ? circleInnerRadius1 : circleInnerRadius2;

					pe.Graphics.DrawLine(pen, circlePoint.X - circleRadius * (float) Math.Sin(angle),
						circlePoint.Y + circleRadius * (float) Math.Cos(angle),
						circlePoint.X - circleInnerRadius * (float) Math.Sin(angle),
						circlePoint.Y + circleInnerRadius * (float) Math.Cos(angle));

					angle += deltaAngle;
				}
			}

			// Arrow

			if (valuePaint != NotAvailableValue)
			{
				GraphicsPath arrowGraphicsPath = new GraphicsPath();

				float valueAngle = GetValueAngle(valuePaint);

				float arrowRadius = circleRadius * arrowRadiusPer / 100;
				float arrowLength = circleRadius * arrowLengthPer / 100;

				arrowGraphicsPath.AddLine(circlePoint.X - arrowRadius, circlePoint.Y,
					circlePoint.X - 1, circlePoint.Y + arrowLength);

				arrowGraphicsPath.AddLine(circlePoint.X - 1, circlePoint.Y + arrowLength,
					circlePoint.X + 1, circlePoint.Y + arrowLength);

				arrowGraphicsPath.AddLine(circlePoint.X + 1, circlePoint.Y + arrowLength,
					circlePoint.X + arrowRadius, circlePoint.Y);

				arrowGraphicsPath.AddArc(circlePoint.X - arrowRadius, circlePoint.Y - arrowRadius,
					2 * arrowRadius, 2 * arrowRadius, 180, 180);

				Matrix matrix = new Matrix();

				matrix.RotateAt(alpha + valueAngle, circlePoint);
				arrowGraphicsPath.Transform(matrix);

				using (Brush brush = new SolidBrush(Color.Red))
				{
					pe.Graphics.FillPath(brush, arrowGraphicsPath);
				}
			}

			using (Brush brush = new SolidBrush(SystemColors.WindowText))
			{
				float circleRadiusHeight = circleRadius * (float) Math.Cos(DegreesToRadians(alpha));
				float circleRadiusWidth = circleRadius * (float) Math.Sin(DegreesToRadians(alpha));

				// Value text

				using (Font valueTextFont = new Font(Font, Font.Style | FontStyle.Bold))
				{
					RectangleF valueTextRectangle = new RectangleF(
						drawRectangle.Left, 
						circlePoint.Y + (circleRadiusHeight - lineHeight) / 2,
						drawRectangle.Width, 
						lineHeight);

					pe.Graphics.DrawString(valueText, valueTextFont, brush, 
						valueTextRectangle, stringFormat);
				}

				// Range text

				float width = Math.Min(drawRectangle.Width / 2 - circleRadiusWidth, circleRadiusWidth / 2);
				float top = circlePoint.Y + circleRadiusHeight + 0.07F * circleRadius;

				RectangleF minRangeTextRectangle = new RectangleF(circlePoint.X - circleRadiusWidth - width,
					top, 2 * width, captionRectangle.Height);

				pe.Graphics.DrawString(minRangeText, Font, brush, minRangeTextRectangle, stringFormat);

				RectangleF maxRangeTextRectangle = new RectangleF(circlePoint.X + circleRadiusWidth - width,
					top, 2 * width, captionRectangle.Height);

				pe.Graphics.DrawString(maxRangeText, Font, brush, maxRangeTextRectangle, stringFormat);

				if (normalText != null)
				{
					RectangleF normalRangeTextRectangle = new RectangleF(
						circlePoint.X - circleRadiusWidth + width, top,
						2 * (circleRadiusWidth - width), captionRectangle.Height);

					pe.Graphics.DrawString(normalText, Font, brush, normalRangeTextRectangle, 
						stringFormat);
				}
			}
		}

		protected override void OnResize(EventArgs e)
		{
			Invalidate();
		}

		protected int GetPointedValue(int value)
		{
			if (value == NotAvailableValue)
				return NotAvailableValue;

			int index1, index2;

			if (value <= points[0].X)
			{
				index1 = 0;
				index2 = 1;
			}
			else if (value >= points[points.Length - 1].X)
			{
				index1 = points.Length - 2;
				index2 = points.Length - 1;
			}
			else
			{
				index1 = 0;
				index2 = points.Length - 1;

				while (index2 - index1 > 1)
				{
					int index = (index1 + index2) / 2;

					Point point = points[index];

					if (value < point.X)
					{
						index2 = index;
					}
					else if (value > point.X)
					{
						index1 = index;
					}
					else
					{
						index1 = index;
						index2 = index + 1;
					}							
				}
			}

			Point point1 = points[index1];
			Point point2 = points[index2];

			return point1.Y + (value - point1.X) * (point2.Y - point1.Y) / (point2.X - point1.X);			
		}

		protected string GetValueText(int value)
		{
			string valueText;

			if (weight != 0)
			{
				string format = weight < 0 ? "F" + (- weight).ToString() : "F0";
				valueText = (value * Math.Pow(10, weight)).ToString(format);
			}
			else
				valueText = value.ToString();

			return valueText;			
		}

		protected float GetValueAngle(int value)
		{
			float valueAngle;

			if (value <= rangePaint.Min)
			{
				valueAngle = 0;
			}
			else if (value >= rangePaint.Max)
			{
				valueAngle = circleAngle;
			}
			else
				valueAngle = (value - rangePaint.Min) * circleAngle / (rangePaint.Max - rangePaint.Min);

			return valueAngle;
		}

		protected double DegreesToRadians(double degrees)
		{
			return degrees * Math.PI / 180;
		}

		// Paint data

		protected Range rangePaint;
		protected string minRangeText;
		protected string maxRangeText;

		protected Range normalRangePaint;
		protected string normalText;

		protected int valuePaint;
		protected string valueText;
		protected bool normalValue;
	}
}
